<?php
session_start();
require_once '../config/db.php';

// Set error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Function to get base URL dynamically
function getBaseUrl() {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $path = dirname(dirname($_SERVER['SCRIPT_NAME']));
    return $protocol . '://' . $host . $path;
}

$error = '';
$success = '';
$step = isset($_GET['step']) ? $_GET['step'] : 'request';
$token = isset($_GET['token']) ? $_GET['token'] : '';

// Get store settings
try {
    $stmt = $pdo->prepare("SELECT setting_key, setting_value FROM store_settings WHERE setting_key IN ('store_name', 'store_logo', 'primary_color', 'secondary_color', 'store_email')");
    $stmt->execute();
    $storeSettings = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    
    $storeName = $storeSettings['store_name'] ?? 'POS System';
    $storeLogo = !empty($storeSettings['store_logo']) ? '../uploads/logos/' . $storeSettings['store_logo'] : '../assets/img/logo.png';
    $primaryColor = $storeSettings['primary_color'] ?? '#007bff';
    $secondaryColor = $storeSettings['secondary_color'] ?? '#0056b3';
    $storeEmail = $storeSettings['store_email'] ?? '';
} catch (PDOException $e) {
    error_log("Error fetching store settings: " . $e->getMessage());
    $storeName = 'POS System';
    $storeLogo = '../assets/img/logo.png';
    $primaryColor = '#007bff';
    $secondaryColor = '#0056b3';
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if ($step === 'request') {
        $email = trim($_POST['email']);
        
        try {
            $stmt = $pdo->prepare('SELECT id, username FROM users WHERE email = ?');
            $stmt->execute([$email]);
    $user = $stmt->fetch();

    if ($user) {
        $token = bin2hex(random_bytes(32));
                $expires = date('Y-m-d H:i:s', strtotime('+1 hour'));
                
                $stmt = $pdo->prepare('INSERT INTO password_resets (user_id, token, expires_at) VALUES (?, ?, ?)');
                $stmt->execute([$user['id'], $token, $expires]);

                $resetLink = getBaseUrl() . "/auth/reset_password.php?step=reset&token=" . $token;
                $to = $email;
                $subject = "Password Reset Request - " . $storeName;
                $message = '<!DOCTYPE html><html><head><meta charset="UTF-8"><title>Password Reset</title></head><body style="font-family:Inter,sans-serif;background:#f8fafc;padding:0;margin:0;">
<div style="max-width:480px;margin:40px auto;background:#fff;border-radius:16px;box-shadow:0 4px 24px #6366f122;padding:32px 28px;">
  <h2 style="color:#6366f1;text-align:center;margin-top:0;">' . htmlspecialchars($storeName) . '</h2>
  <p style="font-size:1.1rem;color:#222;">Hello <b>' . htmlspecialchars($user['username']) . '</b>,</p>
  <p style="font-size:1.05rem;color:#222;">We received a request to reset your password. Click the button below to set a new password. This link will expire in <b>1 hour</b>.</p>
  <div style="text-align:center;margin:32px 0;">
    <a href="' . $resetLink . '" style="display:inline-block;padding:14px 32px;background:linear-gradient(90deg,#6366f1 60%,#38bdf8 100%);color:#fff;font-weight:600;font-size:1.1rem;border-radius:8px;text-decoration:none;">Reset Password</a>
  </div>
  <p style="font-size:1rem;color:#444;">If the button above does not work, copy and paste this link into your browser:</p>
  <div style="word-break:break-all;font-size:0.97rem;background:#f3f4f6;padding:10px 12px;border-radius:6px;margin-bottom:24px;">' . $resetLink . '</div>
  <p style="font-size:1rem;color:#888;">If you did not request this reset, you can safely ignore this email.</p>
  <hr style="border:none;border-top:1px solid #e0e7ff;margin:32px 0 16px 0;">
  <div style="font-size:0.95rem;color:#888;text-align:center;">&copy; ' . date('Y') . ' ' . htmlspecialchars($storeName) . '. All rights reserved.</div>
</div></body></html>';
                
                // Use sendEmail() for HTML email
                require_once '../includes/email_functions.php';
                if (sendEmail($to, $subject, $message, [])) { // Empty array for attachments
                    $success = 'Password reset instructions have been sent to your email.';
                } else {
                    $error = 'Failed to send reset email. Please try again.';
                }
            } else {
                $error = 'Email address not found in our records. Please check your email and try again.';
            }
        } catch (PDOException $e) {
            error_log("Database error: " . $e->getMessage());
            $error = 'An error occurred. Please try again later.';
        }
    } elseif ($step === 'reset') {
        // Debug: Log the token being processed
        error_log("Reset password step - Token: " . $token);
        
        if (empty($token)) {
            $error = 'No reset token provided.';
        } else {
            // Check if token exists and is valid
            try {
                $stmt = $pdo->prepare('SELECT pr.user_id, u.email, pr.expires_at FROM password_resets pr JOIN users u ON pr.user_id = u.id WHERE pr.token = ? AND pr.expires_at > NOW() AND pr.used = 0');
                $stmt->execute([$token]);
                $reset = $stmt->fetch();
                
                if (!$reset) {
                    $error = 'Invalid or expired reset token. Please request a new password reset.';
                }
            } catch (PDOException $e) {
                error_log("Database error in reset step: " . $e->getMessage());
                $error = 'Database error occurred. Please try again.';
            }
        }
        
        // Only process form submission if there's no error
        if (empty($error) && $_SERVER['REQUEST_METHOD'] === 'POST') {
        $password = $_POST['password'];
        $confirm_password = $_POST['confirm_password'];

        if ($password !== $confirm_password) {
            $error = 'Passwords do not match.';
        } elseif (strlen($password) < 8) {
            $error = 'Password must be at least 8 characters long.';
        } else {
            try {
                $stmt = $pdo->prepare('SELECT pr.user_id, u.email FROM password_resets pr JOIN users u ON pr.user_id = u.id WHERE pr.token = ? AND pr.expires_at > NOW() AND pr.used = 0');
            $stmt->execute([$token]);
            $reset = $stmt->fetch();

            if ($reset) {
                    $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                    $stmt = $pdo->prepare('UPDATE users SET password = ? WHERE id = ?');
                        $stmt->execute([$hashed_password, $reset['user_id']]);

                    $stmt = $pdo->prepare('UPDATE password_resets SET used = 1 WHERE token = ?');
                        $stmt->execute([$token]);
                    
                    $stmt = $pdo->prepare('INSERT INTO system_logs (user_id, action, details) VALUES (?, ?, ?)');
                    $stmt->execute([$reset['user_id'], 'password_reset', 'Password reset successful']);

                        $success = 'Your password has been reset successfully. You can now login with your new password.';
                    header('refresh:3;url=login.php');
                } else {
                    $error = 'Invalid or expired reset token.';
                }
            } catch (PDOException $e) {
                error_log("Database error: " . $e->getMessage());
                $error = 'An error occurred. Please try again later.';
                }
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reset Password - <?php echo htmlspecialchars($storeName); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <style>
        :root {
            --primary-color: <?php echo $primaryColor; ?>;
            --secondary-color: <?php echo $secondaryColor; ?>;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0;
            padding: 20px;
        }
        
        .reset-container {
            width: 100%;
            max-width: 420px;
            margin: 0 auto;
        }
        
        .card {
            background: rgba(255, 255, 255, 0.9);
            backdrop-filter: blur(10px);
            border: none;
            border-radius: 20px;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            overflow: hidden;
            transition: transform 0.3s ease;
        }
        
        .card:hover {
            transform: translateY(-5px);
        }
        
        .card-header {
            background: var(--primary-color);
            color: white;
            text-align: center;
            padding: 30px 20px;
            position: relative;
            padding-top: 80px;
        }
        
        .profile-photo {
            width: 100px;
            height: 100px;
            border-radius: 50%;
            border: 4px solid white;
            position: absolute;
            top: -50px;
            left: 50%;
            transform: translateX(-50%);
            background-color: white;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.15);
            overflow: hidden;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: transform 0.3s ease;
        }
        
        .profile-photo:hover {
            transform: translateX(-50%) scale(1.05);
        }
        
        .profile-photo img {
            width: 100%;
            height: 100%;
            object-fit: contain;
            padding: 10px;
        }
        
        .card-header h4 {
            margin: 0;
            font-size: 1.5rem;
            font-weight: 600;
            letter-spacing: 0.5px;
        }
        
        .card-body {
            padding: 30px;
        }
        
        .form-control {
            border-radius: 10px;
            padding: 12px 15px;
            border: 2px solid #e1e1e1;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            border-color: var(--primary-color);
            box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.15);
        }
        
        .input-group-text {
            border-radius: 10px;
            border: 2px solid #e1e1e1;
            background: white;
        }
        
        .btn-primary {
            background: var(--primary-color);
            border: none;
            border-radius: 10px;
            padding: 12px;
            font-weight: 600;
            letter-spacing: 0.5px;
            transition: all 0.3s ease;
        }
        
        .btn-primary:hover {
            background: var(--secondary-color);
            transform: translateY(-2px);
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
        }
        
        .btn-primary:active {
            transform: translateY(0);
        }
        
        .alert {
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 20px;
            border: none;
        }
        
        .alert-danger {
            background: #fee2e2;
            color: #dc2626;
        }
        
        .alert-success {
            background: #dcfce7;
            color: #16a34a;
        }
        
        .back-to-login {
            color: var(--primary-color);
            text-decoration: none;
            font-weight: 500;
            transition: color 0.3s ease;
            display: inline-block;
            margin-top: 15px;
        }
        
        .back-to-login:hover {
            color: var(--secondary-color);
        }
        
        .password-strength {
            height: 5px;
            margin-top: 10px;
            border-radius: 5px;
            transition: all 0.3s ease;
        }
        
        .password-strength.weak {
            background: #dc2626;
            width: 33.33%;
        }
        
        .password-strength.medium {
            background: #f59e0b;
            width: 66.66%;
        }
        
        .password-strength.strong {
            background: #16a34a;
            width: 100%;
        }
        
        .password-requirements {
            font-size: 0.85rem;
            color: #666;
            margin-top: 10px;
        }
        
        .password-requirements ul {
            list-style: none;
            padding-left: 0;
            margin-bottom: 0;
        }
        
        .password-requirements li {
            margin-bottom: 5px;
        }
        
        .password-requirements li i {
            margin-right: 5px;
        }
        
        .password-requirements li.valid {
            color: #16a34a;
        }
        
        .password-requirements li.invalid {
            color: #dc2626;
        }
        
        @media (max-width: 576px) {
            .reset-container {
                max-width: 100%;
            }
            
            .card-header {
                padding: 20px;
                padding-top: 70px;
            }
            
            .profile-photo {
                width: 80px;
                height: 80px;
                top: -40px;
            }
            
            .card-body {
                padding: 20px;
            }
        }
        
        .loading {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(255, 255, 255, 0.8);
            backdrop-filter: blur(5px);
            z-index: 9999;
            justify-content: center;
            align-items: center;
        }
        
        .loading-spinner {
            width: 50px;
            height: 50px;
            border: 5px solid #f3f3f3;
            border-top: 5px solid var(--primary-color);
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="loading">
        <div class="loading-spinner"></div>
    </div>
    
    <div class="reset-container">
        <div class="card">
            <div class="card-header">
                <div class="profile-photo">
                    <img src="<?php echo htmlspecialchars($storeLogo); ?>" alt="<?php echo htmlspecialchars($storeName); ?> Logo">
                </div>
                <h4><?php echo $step === 'request' ? 'Reset Password' : 'Set New Password'; ?></h4>
            </div>
            <div class="card-body">
            <?php if ($error): ?>
                    <div class="alert alert-danger">
                        <i class="fas fa-exclamation-circle me-2"></i><?php echo htmlspecialchars($error); ?>
                    </div>
            <?php endif; ?>

                <?php if ($success): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle me-2"></i><?php echo htmlspecialchars($success); ?>
                    </div>
                <?php endif; ?>
                
                <?php if ($step === 'request'): ?>
                    <form method="post" id="resetForm">
                        <div class="mb-4">
                            <label for="email" class="form-label">Email Address</label>
                            <div class="input-group">
                                <span class="input-group-text">
                                    <i class="fas fa-envelope"></i>
                                </span>
                                <input type="email" class="form-control" id="email" name="email" required 
                                       placeholder="Enter your email address" autocomplete="email">
                            </div>
                        </div>
                        
                        <button type="submit" class="btn btn-primary w-100 mb-3">
                            <i class="fas fa-paper-plane me-2"></i>Send Reset Link
                        </button>
                </form>
                <?php elseif ($step === 'reset'): ?>
                    <form method="post" id="resetForm">
                        <div class="mb-4">
                            <label for="password" class="form-label">New Password</label>
                            <div class="input-group">
                                <span class="input-group-text">
                                    <i class="fas fa-lock"></i>
                                </span>
                                <input type="password" class="form-control" id="password" name="password" required 
                                       placeholder="Enter new password" autocomplete="new-password">
                                <button class="btn btn-outline-secondary" type="button" id="togglePassword">
                                    <i class="fas fa-eye"></i>
                                </button>
                            </div>
                            <div class="password-strength"></div>
                            <div class="password-requirements">
                                <ul>
                                    <li id="length"><i class="fas fa-times"></i> At least 8 characters</li>
                                    <li id="uppercase"><i class="fas fa-times"></i> One uppercase letter</li>
                                    <li id="lowercase"><i class="fas fa-times"></i> One lowercase letter</li>
                                    <li id="number"><i class="fas fa-times"></i> One number</li>
                                    <li id="special"><i class="fas fa-times"></i> One special character</li>
                                </ul>
                            </div>
                        </div>
                        
                        <div class="mb-4">
                            <label for="confirm_password" class="form-label">Confirm Password</label>
                            <div class="input-group">
                                <span class="input-group-text">
                                    <i class="fas fa-lock"></i>
                                </span>
                                <input type="password" class="form-control" id="confirm_password" name="confirm_password" required 
                                       placeholder="Confirm new password" autocomplete="new-password">
                    </div>
                    </div>
                        
                        <button type="submit" class="btn btn-primary w-100 mb-3">
                            <i class="fas fa-key me-2"></i>Reset Password
                        </button>
                </form>
            <?php endif; ?>
                
                <div class="text-center">
                    <a href="login.php" class="back-to-login">
                        <i class="fas fa-arrow-left me-1"></i>Back to Login
                    </a>
                </div>
            </div>
        </div>
        
        <div class="footer mt-4 text-center text-muted">
        <?php include '../includes/footer.php'; ?>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const form = document.getElementById('resetForm');
            const loading = document.querySelector('.loading');
            const password = document.getElementById('password');
            const confirmPassword = document.getElementById('confirm_password');
            const togglePassword = document.getElementById('togglePassword');
            const strengthBar = document.querySelector('.password-strength');
            const requirements = {
                length: document.getElementById('length'),
                uppercase: document.getElementById('uppercase'),
                lowercase: document.getElementById('lowercase'),
                number: document.getElementById('number'),
                special: document.getElementById('special')
            };
            
            if (togglePassword) {
                togglePassword.addEventListener('click', function() {
                    const type = password.getAttribute('type') === 'password' ? 'text' : 'password';
                    password.setAttribute('type', type);
                    this.querySelector('i').classList.toggle('fa-eye');
                    this.querySelector('i').classList.toggle('fa-eye-slash');
                });
            }
            
            if (password) {
                password.addEventListener('input', function() {
                    const value = this.value;
                    let strength = 0;
                    
                    // Check length
                    if (value.length >= 8) {
                        strength++;
                        requirements.length.classList.add('valid');
                        requirements.length.classList.remove('invalid');
                        requirements.length.querySelector('i').className = 'fas fa-check';
                    } else {
                        requirements.length.classList.remove('valid');
                        requirements.length.classList.add('invalid');
                        requirements.length.querySelector('i').className = 'fas fa-times';
                    }
                    
                    // Check uppercase
                    if (/[A-Z]/.test(value)) {
                        strength++;
                        requirements.uppercase.classList.add('valid');
                        requirements.uppercase.classList.remove('invalid');
                        requirements.uppercase.querySelector('i').className = 'fas fa-check';
                    } else {
                        requirements.uppercase.classList.remove('valid');
                        requirements.uppercase.classList.add('invalid');
                        requirements.uppercase.querySelector('i').className = 'fas fa-times';
                    }
                    
                    // Check lowercase
                    if (/[a-z]/.test(value)) {
                        strength++;
                        requirements.lowercase.classList.add('valid');
                        requirements.lowercase.classList.remove('invalid');
                        requirements.lowercase.querySelector('i').className = 'fas fa-check';
                    } else {
                        requirements.lowercase.classList.remove('valid');
                        requirements.lowercase.classList.add('invalid');
                        requirements.lowercase.querySelector('i').className = 'fas fa-times';
                    }
                    
                    // Check number
                    if (/[0-9]/.test(value)) {
                        strength++;
                        requirements.number.classList.add('valid');
                        requirements.number.classList.remove('invalid');
                        requirements.number.querySelector('i').className = 'fas fa-check';
                    } else {
                        requirements.number.classList.remove('valid');
                        requirements.number.classList.add('invalid');
                        requirements.number.querySelector('i').className = 'fas fa-times';
                    }
                    
                    // Check special character
                    if (/[^A-Za-z0-9]/.test(value)) {
                        strength++;
                        requirements.special.classList.add('valid');
                        requirements.special.classList.remove('invalid');
                        requirements.special.querySelector('i').className = 'fas fa-check';
                    } else {
                        requirements.special.classList.remove('valid');
                        requirements.special.classList.add('invalid');
                        requirements.special.querySelector('i').className = 'fas fa-times';
                    }
                    
                    // Update strength bar
                    strengthBar.className = 'password-strength';
                    if (strength <= 2) {
                        strengthBar.classList.add('weak');
                    } else if (strength <= 4) {
                        strengthBar.classList.add('medium');
                    } else {
                        strengthBar.classList.add('strong');
                    }
                });
            }
            
            form.addEventListener('submit', function() {
                if (password && confirmPassword) {
                    if (password.value !== confirmPassword.value) {
                        alert('Passwords do not match!');
                        return false;
                    }
                }
                loading.style.display = 'flex';
            });
        });
    </script>
</body>
</html>
